package com.yaskawa.demoapp;

// Network related imports
import java.nio.ByteOrder;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.DataOutputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.net.Socket;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.net.SocketTimeoutException;

// Droid imports
import android.app.Activity;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;

// General imports
import java.lang.Float;
import java.lang.Integer;
import java.lang.StringBuffer;

public class demo_app extends Activity
{
	// View elements
	private EditText input_controller_ip;
	private Button button_toggle_connection;
	private TextView output_velocity, output_position, output_alarmid, output_error;
	
	// Connection globals
	private boolean socket_connected = false;
	private InetSocketAddress sockaddr = null;
	private Socket socket = null;
	
	// I/O streams
	private DataOutputStream output_stream = null;
	private DataInputStream input_stream = null;
	
	// Handler for the packet listener thread
	private Handler packet_listen_handler;
	
    @Override
    public void onCreate(Bundle savedInstanceState)
    {
		super.onCreate( savedInstanceState );
		setContentView( R.layout.main );
		
		// Initialize all views
		this.find_all_views_by_id();
		output_error.setText( ByteOrder.nativeOrder().toString() );
		
		// Set up UI elements
		button_toggle_connection.setOnClickListener( toggle_connection_button_listener );
		
		// Set up packet listening thread
		packet_listen_handler = new Handler();
    }
	
	private void find_all_views_by_id()
	{
		input_controller_ip = (EditText) findViewById( R.id.input_controller_ip );
		button_toggle_connection = (Button) findViewById( R.id.button_toggle_connection );
		output_velocity = (TextView) findViewById( R.id.output_velocity );
		output_position = (TextView) findViewById( R.id.output_position );
		output_alarmid = (TextView) findViewById( R.id.output_alarmid );
		output_error = (TextView) findViewById( R.id.output_error );
	}
	
	// Socket Manager Listener
	private Button.OnClickListener toggle_connection_button_listener = new Button.OnClickListener()
	{
		@Override
		public void onClick( View v )
		{
			if ( socket_connected == false )
			{
				// Try to open a connection
				String ip = input_controller_ip.getText().toString();
				if ( ip.isEmpty() == false )
				{
					connect_socket( ip );
				}
			}
			else
			{
				// Try to close the connection
				disconnect_socket();
			}
		}
	};
	
	public void clearError( View v )
	{
		output_error.setText( "" );
	}
	
	private Runnable get_controller_data_packet = new Runnable()
	{
		int alarm;
		float velocity, position;
		
		@Override
		public void run()
		{
			if ( send_command( 40 ) )
			{
				try {
					velocity = input_stream.readFloat();
					position = input_stream.readFloat();
					alarm = input_stream.readShort();
					
					output_velocity.setText( String.format( "%.4f", velocity ) );
					output_position.setText( String.format( "%.4f", position ) );
					output_alarmid.setText( String.format( "%X", alarm ) );
				} catch ( UnknownHostException e ) {
					output_error.setText( "Unknown host." );
				} catch ( IOException e ) {
					output_error.setText( "IOException: " + e.toString() );
				}
			}
			else
			{
				output_error.setText( "Couldn't send Get Packet command" );
				return;
			}
			
			// Update every 0.2s
			packet_listen_handler.postDelayed( get_controller_data_packet, 200 );
		}
	};
	
	void start_servo_packet_read(){ get_controller_data_packet.run(); }
	void stop_servo_packet_read(){ packet_listen_handler.removeCallbacks( get_controller_data_packet ); }
	
	// Send Stay Alive (SA) command every 2s so that the controller knows the app is still
	//   connected.
	private Runnable keep_connection_alive = new Runnable()
	{
		@Override
		public void run()
		{
			if ( !send_command( 1 ) )
			{
				output_error.setText( "Couldn't send stay alive command." );
			}
			
			// Update every 2.0s
			packet_listen_handler.postDelayed( keep_connection_alive, 2000 );
		}
	};
	void start_stay_alive(){ keep_connection_alive.run(); }
	void stop_stay_alive(){ packet_listen_handler.removeCallbacks( keep_connection_alive ); }
	
	/* Connection methods */
	// Connect
	private void connect_socket( final String local_ip )
	{
		if ( socket_connected == false )
		{
			try {
				// Create the connection, set a 4s timeout
				sockaddr = new InetSocketAddress( local_ip, 23 );
				socket = new Socket();
				// Set 2s timeout for IO, throws "SocketTimeoutException"
				socket.setSoTimeout( 2000 );
				socket.connect( sockaddr, 4000 );
				
				if ( socket.isConnected() == true )
				{
					// Create input and output streams
					try {
						output_stream = new DataOutputStream( socket.getOutputStream() );
						input_stream = new DataInputStream( socket.getInputStream() );
					} catch( Throwable t ) {
						output_error.setText( "Error in creating io stream" );
					}
					
					// Update socket_connected and change view items
					socket_connected = true;
					button_toggle_connection.setText( "Disconnect" );
					
					// Enable reading from device
					start_servo_packet_read();
					
					// Start SA (stay-alive) service
					start_stay_alive();
				}
				else
				{
					output_error.setText( "Socket not connected." );
				}
			} catch ( UnknownHostException e ) {
				output_error.setText( "Unknown host." );
			} catch ( SocketTimeoutException e ) {
				output_error.setText( "Socket timed out" );
			} catch ( IOException e ) {
				output_error.setText( "IOException: " + e.toString() );
			}
		}
	}
	
	// Disconnect
	private void disconnect_socket()
	{
		if ( socket_connected == true )
		{
			// Close servo reading
			stop_servo_packet_read();
			
			// Stop SA
			stop_stay_alive();
			
			// Tell the controller that the app is disconnecting
			if ( send_command( 99 ) == false )
			{
				// Server side disconnect failed
			}
			
			// Try to close the output stream
			if ( output_stream != null )
			{
				try {
					output_stream.flush();
					output_stream.close();
				} catch ( IOException e ) {
					output_error.setText( "Eerror flushing/closing output stream." );
				}
			}
			
			// Try to close the input stream
			if ( input_stream != null )
			{
				try {
					input_stream.close();
				} catch ( IOException e ) {
					output_error.setText( "Error closing input stream." );
				}
			}
			
			// Try to close the socket
			if ( socket != null )
			{
				try {
					socket.close();
					
					// Make sure the socket closed
					if ( socket.isClosed() == true )
					{
						// Update socket_connected and view itesm
						socket_connected = false;
						button_toggle_connection.setText( "Connect" );
					}
				} catch ( IOException e ) {
					output_error.setText( "IOException: " + e.toString() );
				}
			}
		}
	}
	
	private boolean send_command( int command )
	{
		if ( socket_connected == true )
		{
			try {
				output_stream.writeByte( command );
				
				return true;
				// Do things associated with this command
			} catch ( UnknownHostException e ) {
				// Unknown Host
				output_error.setText( "Unknown host." );
			} catch ( IOException e ) {
				// IOException
				output_error.setText( "IOException: " + e.toString() );
			} catch ( Throwable t ) {
				// Some other error
				output_error.setText( "Some other error: " + t.toString() );
				return false;
			}
		}
		else
		{
			// No controller connected
			output_error.setText( "No controller connected." );
			return false;
		}
		
		return false;
	}
}
